﻿#include  "StdAfx.h"

#include  "szArchiveEnumeratorImpl.hpp"
#include  <szArchiveOpenClientCallback.hpp>
#include  <szGetPasswordClientCallback.hpp>
#include  <szCodecManager.hpp>
#include  <szArchiveVolume.hpp>
#include  <szArchiveContent.hpp>
#include  <szStoredItem.hpp>
#include  <szRuntimeException.hpp>
#include  <szPath.hpp>
#include  <7zip/PropID.h>
#include  <7zip/Archive/IArchive.h>
#include  <7zip/UI/Common/LoadCodecs.h>
#include  "szPropertyConversions.hpp"

#ifndef RNINOK // Return Null If Not OK
#define RNINOK(x) { int __result__ = (x); if (__result__ != 0) return 0; }
#endif

SZ_NS_BEG(szpp)

ArchiveEnumeratorImpl::ArchiveEnumeratorImpl(ArchiveVolume *volume, u64 searchLimitSize, ArchiveOpenClientCallback *open, GetPasswordClientCallback *pwd)
: volume(volume), searchLimitSize(searchLimitSize), openCallback(open), pwdCallback(pwd), filePropertyInfo(), archive()
{
}

ArchiveEnumeratorImpl::~ArchiveEnumeratorImpl()
{
}

ArchiveContent *ArchiveEnumeratorImpl::EnumerateUnknown(ArchiveHandler **handler)
{
  assert(handler != 0);

  for (int i = 0; i < CodecManager::GetCodecs()->Formats.Size(); ++i)
  {
    const ArchiveHandler &currentHandler = CodecManager::GetHandler(i);
    ArchiveContent *content = Enumerate(&currentHandler);
    if (0 != content)
    {
      *handler = currentHandler.Clone();
      return content;
    }
  }
  return 0;
}

ArchiveContent *ArchiveEnumeratorImpl::Enumerate(const ArchiveHandler *handler)
{
  CodecManager::GetCodecs()->CreateInArchive(handler->GetIndex(), archive);
  if (0 == archive)
    BOOST_THROW_EXCEPTION(RuntimeException(SZT("Cannot create handler")));

#ifdef EXTERNAL_CODECS
  {
    // コーダーの情報を入力アーカイブハンドラに設定（ハンドラは内部で必要な情報をコピーする）。
    CMyComPtr<ISetCompressCodecsInfo> setCompressCodecsInfo;
    archive.QueryInterface(IID_ISetCompressCodecsInfo, (void **)&setCompressCodecsInfo);
    if (setCompressCodecsInfo)
      RNINOK(setCompressCodecsInfo->SetCompressCodecsInfo(CodecManager::GetCodecs()));
  }
#endif

  try
  {
    CMyComPtr<IInStream> stream;
    HRESULT hr = volume->GetArchiveStream(&stream);
    if (SUCCEEDED(hr))
    {
      hr = archive->Open(stream, &searchLimitSize, this);
      if (S_OK == hr) // エラーは起きなかったがオープンできなかった場合などに S_FALSE が返されることがあるので単に SUCCEEDED を使うとまずい
      {
        std::auto_ptr<ArchiveContent> content(new ArchiveContent());

        // ファイルプロパティ ID の一覧を管理
        UInt32 numProps = 0;
        if (S_OK != archive->GetNumberOfProperties(&numProps))
          return 0;

        // ファイルプロパティ一覧を取得
        CMyComBSTR name;
        PROPID     propID;
        VARTYPE    varType;
        for (UInt32 i = 0; i < numProps; ++i)
          if (S_OK == archive->GetPropertyInfo(i, &name, &propID, &varType))
            filePropertyInfo.AddProperty(propID);

        UInt32 numItems = 0;
        if (S_OK != archive->GetNumberOfItems(&numItems))
          return 0;

        if (openCallback)
          openCallback->SetTotal(numItems, 0);

        // 格納アイテムの情報を取得（アーカイブプロパティとして圧縮サイズと非圧縮サイズが得られなかったときのためにそれらの概算値も計算しておく）
        u64 unpSize = 0, pakSize = 0;
        for (index_t i = 0; i < numItems; ++i)
        {
          StoredItem *item = CreateStoredItem(i);
          unpSize += item->GetActualSize();
          pakSize += item->GetStoredSize();
          content->AddItem(item);

          if (openCallback)
            if (E_ABORT == openCallback->SetCompleted(i, 0))
              return 0;
        }

        content->ConstructTree(ExtractStem(volume->GetVirtualPath()));

        // 格納ファイルの情報は取得できたのでアーカイブプロパティを設定する
        {
          ArchiveProperty archiveProperty;

          {
            // 取得できるならストリームサイズを圧縮サイズに設定しておく
            CMyComPtr<IStreamGetSize> streamGetSize;
            hr = stream->QueryInterface(IID_IStreamGetSize, (void **)&streamGetSize);
            if (S_OK == hr)
            {
              UInt64 size;
              if (S_OK == streamGetSize->GetSize(&size))
                pakSize = size;
            }
          }

          // gz, Z などの場合は非圧縮サイズがまったく分からないので 0 になっている。このままでもいいが、とりあえず圧縮サイズと同じ値にしておく。
          if (0 == unpSize)
            unpSize = pakSize;

          // 格納アイテム数、圧縮サイズ、非圧縮サイズの概算値を設定
          archiveProperty.SetNumberOfItems(content->GetNumberOfItems());
          archiveProperty.SetCompressedSize(pakSize);
          archiveProperty.SetUncompressedSize(unpSize);

          // その他のアーカイブプロパティを設定
          FillArchiveProperty(&archiveProperty);
          content->SetArchiveProperty(archiveProperty);
        }

        return content.release();
      }
    }
  }
  catch (RuntimeException &)
  {
    throw;
  }
  catch (...)
  {
  }

  return 0;
}

ArchiveContent *ArchiveEnumeratorImpl::EnumerateUnknown(ArchiveHandler **handler, ArchiveVolume *volume, u64 searchLimitSize, ArchiveOpenClientCallback *open, GetPasswordClientCallback *pwd)
{
  CMyComPtr<ArchiveEnumeratorImpl> impl(new ArchiveEnumeratorImpl(volume, searchLimitSize, open, pwd));
  return impl->EnumerateUnknown(handler);
}

ArchiveContent *ArchiveEnumeratorImpl::Enumerate(const ArchiveHandler *handler, ArchiveVolume *volume, u64 searchLimitSize, ArchiveOpenClientCallback *open, GetPasswordClientCallback *pwd)
{
  CMyComPtr<ArchiveEnumeratorImpl> impl(new ArchiveEnumeratorImpl(volume, searchLimitSize, open, pwd));
  return impl->Enumerate(handler);
}

/// <todo>もっと多くの属性に（真面目に）対応。</todo>
StoredItem *ArchiveEnumeratorImpl::CreateStoredItem(index_t index)
{
  PROPID id;
  std::auto_ptr<StoredItem> item(new StoredItem(index));

  // ファイル名（パス）
  if (0 != (id = filePropertyInfo.HasPath()))
    item->SetName(GetStringProperty(archive, index, id));

  // 展開サイズ
  if (0 != (id = filePropertyInfo.HasActualSize()))
    item->SetActualSize(GetU64Property(archive, index, id));

  // 格納サイズ
  if (0 != (id = filePropertyInfo.HasStoredSize()))
    item->SetStoredSize(GetU64Property(archive, index, id));

  // 属性
  if (0 != (id = filePropertyInfo.HasAttributes()))
    item->SetAttributes(GetU32Property(archive, index, id));

  // 作成日時
  if (0 != (id = filePropertyInfo.HasCreated()))
    item->SetCreated(GetTimeProperty(archive, index, id));

  // アクセス日時
  if (0 != (id = filePropertyInfo.HasAccessed()))
    item->SetAccessed(GetTimeProperty(archive, index, id));

  // 変更日時
  if (0 != (id = filePropertyInfo.HasModified()))
    item->SetModified(GetTimeProperty(archive, index, id));

  // ディレクトリ
  if (0 != (id = filePropertyInfo.HasIsDir()))
    item->SetAttributes(GetBoolProperty(archive, index, id) ? FILE_ATTRIBUTE_DIRECTORY : 0);

  // もっと多くの属性に対応。

  return item.release();
}

/// <todo>もっと多くの属性に（真面目に）対応。</todo>
void ArchiveEnumeratorImpl::FillArchiveProperty(ArchiveProperty *archiveProperty)
{
  UInt32 numProps;
  if (S_OK == archive->GetNumberOfArchiveProperties(&numProps))
  {
    for (UInt32 j = 0; j < numProps; ++j)
    {
      using namespace NWindows::NCOM;

      CMyComBSTR name;
      PROPID propID;
      VARTYPE vt;
      if (archive->GetArchivePropertyInfo(j, &name, &propID, &vt) != S_OK)
        continue;
      CPropVariant prop;
      if (archive->GetArchiveProperty(propID, &prop) != S_OK)
        continue;

      // 7-Zip が管理しているアーカイブプロパティを SevenZip++ 形式に変換
      switch (propID)
      {
      case kpidSize:
        archiveProperty->SetCompressedSize(ConvertPropVariantToU64(prop));
        break;
      case kpidPackSize:
        archiveProperty->SetUncompressedSize(ConvertPropVariantToU64(prop));
        break;
      case kpidSolid:
        archiveProperty->SetIsSolid(ConvertPropVariantToBool(prop));
        break;
      case kpidIsVolume:
        archiveProperty->SetIsMultivolume(ConvertPropVariantToBool(prop));
        break;
      case kpidNumVolumes:
        // 7-Zip はマルチボリュームの認識は甘いので、RAR の途中のボリュームを開くとボリューム数として 1 を返してくる（先頭ボリュームを開けば正しく処理できるが）。
        archiveProperty->SetNumberOfVolumes(ConvertPropVariantToU32(prop));
        break;
    
      // もっと多くの属性に対応。
      }
    }
  }
}

STDMETHODIMP ArchiveEnumeratorImpl::SetTotal(const UInt64 *files, const UInt64 *bytes)
{
  if (openCallback)
  {
    const u64 nFiles = (files ? *files : 0);
    const u64 nBytes = (bytes ? *bytes : 0);
    return openCallback->SetTotal(nFiles, nBytes);
  }

  return S_OK;
}

STDMETHODIMP ArchiveEnumeratorImpl::SetCompleted(const UInt64 *files, const UInt64 *bytes)
{
  if (openCallback)
  {
    const u64 nFiles = (files ? *files : 0);
    const u64 nBytes = (bytes ? *bytes : 0);
    return openCallback->SetCompleted(nFiles, nBytes);
  }

  return S_OK;
}

STDMETHODIMP  ArchiveEnumeratorImpl::GetProperty(PROPID propID, PROPVARIANT *value)
{
  return E_FAIL;
}

STDMETHODIMP  ArchiveEnumeratorImpl::GetStream(const wchar_t *name, IInStream **inStream)
{
  return S_FALSE;
}

STDMETHODIMP  ArchiveEnumeratorImpl::SetSubArchiveName(const wchar_t *name)
{
//  _subArchiveMode = true;
//  _subArchiveName = name;
//  TotalSize = 0;
//  return  S_OK;
  return S_OK;
}

#ifndef _NO_CRYPTO

STDMETHODIMP ArchiveEnumeratorImpl::CryptoGetTextPassword(BSTR *password)
{
  // コールバックが設定すらされていない場合は S_FALSE を返すようにしている
  HRESULT hr = S_FALSE;
  if (pwdCallback)
  {
    szstring clientPassword;
    if (SUCCEEDED(hr = pwdCallback->GetPassword(&clientPassword)))
      hr = StringToBstr(clientPassword.c_str(), password);
  }
  return hr;
}

#endif

SZ_NS_END(szpp)
